import { NextResponse } from 'next/server';
import { sendServiceNotificationEmail } from '@/lib/email';
import dbConnect from '@/lib/db';
import Service from '@/lib/models/Service';

export async function POST(request) {
  try {
    
    await dbConnect();
    
    const body = await request.json();
    const { serviceName, name, email, phone, message } = body;

    if (!serviceName || !name || !email || !phone || !message) {
      return NextResponse.json(
        { 
          success: false,
          error: 'All fields are required' 
        },
        { status: 400 }
      );
    }

    const emailRegex = /^\w+([.-]?\w+)*@\w+([.-]?\w+)*(\.\w{2,3})+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { 
          success: false,
          error: 'Please enter a valid email address' 
        },
        { status: 400 }
      );
    }

    const phoneRegex = /^\+?[1-9]\d{1,14}$/;
    if (!phoneRegex.test(phone.replace(/\s/g, ''))) {
      return NextResponse.json(
        { 
          success: false,
          error: 'Please enter a valid phone number' 
        },
        { status: 400 }
      );
    }

    const service = await Service.create({
      serviceName: serviceName.trim(),
      name: name.trim(),
      email: email.toLowerCase().trim(),
      phone: phone.trim(),
      message: message.trim(),
    });
    await sendServiceNotificationEmail(service);

    return NextResponse.json(
      { 
        success: true,
        message: 'Service submission received successfully',
        data: service 
      },
      { status: 201 }
    );
  } catch (error) {
    console.error('Create service error:', error);
    
    if (error.name === 'ValidationError') {
      const errors = Object.values(error.errors).map(err => err.message);
      return NextResponse.json(
        { 
          success: false,
          error: 'Validation failed',
          details: errors 
        },
        { status: 400 }
      );
    }
    
    return NextResponse.json(
      { 
        success: false,
        error: 'Failed to submit service form' 
      },
      { status: 500 }
    );
  }
}